import "dotenv/config";
import express from "express";
import helmet from "helmet";
import rateLimit from "express-rate-limit";
import { DateTime } from "luxon";
import { createCpanelAccount, removeCpanelAccount } from "./whm.js";
import { insertOrder, listOrders, getExpiredUndeletedOrders, markDeleted } from "./db.js";

const app = express();

app.use(helmet());
app.use(express.json({ limit: "200kb" }));
app.use(express.static("public"));

app.set("trust proxy", 1);

const orderLimiter = rateLimit({
  windowMs: 60 * 1000,
  limit: 10,
  standardHeaders: true,
  legacyHeaders: false
});

function getPackages() {
  const raw = process.env.PACKAGES || "";
  return raw.split(",").map(s => s.trim()).filter(Boolean);
}

function isValidDomain(domain) {
  return /^[a-z0-9.-]+\.[a-z]{2,}$/i.test(domain) && domain.length <= 253;
}
function isValidUsername(u) {
  return /^[a-z][a-z0-9]{0,15}$/i.test(u);
}
function isValidEmail(e) {
  return /^[^\s@]+@[^\s@]+\.[^\s@]+$/.test(e);
}
function isStrongPassword(p) {
  return typeof p === "string"
    && p.length >= 10
    && /[a-z]/.test(p)
    && /[A-Z]/.test(p)
    && /[0-9]/.test(p);
}

function parseWhmCreateResult(whm) {
  // createacct biasanya: whm.data.result[0].status + statusmsg
  const r0 = whm?.data?.result?.[0];
  if (r0 && typeof r0.status !== "undefined") {
    const ok = Number(r0.status) === 1;
    return { ok, message: r0.statusmsg || (ok ? "Account created." : "Failed") };
  }
  // fallback
  if (whm?.metadata && typeof whm.metadata.result !== "undefined") {
    const ok = Number(whm.metadata.result) === 1;
    return { ok, message: whm.metadata.reason || (ok ? "Account created." : "Failed") };
  }
  return { ok: false, message: "Unknown WHM response." };
}

function parseWhmDeleteResult(whm) {
  if (whm?.metadata && typeof whm.metadata.result !== "undefined") {
    const ok = Number(whm.metadata.result) === 1;
    return { ok, message: whm.metadata.reason || (ok ? "Account removed." : "Remove failed") };
  }
  return { ok: false, message: "Unknown WHM delete response." };
}

// --- API: packages
app.get("/api/packages", (req, res) => {
  res.json({ ok: true, packages: getPackages() });
});

// --- API: create order + create cPanel
app.post("/api/order", orderLimiter, async (req, res) => {
  try {
    const {
      domain,
      username,
      password,
      email,
      pkg,
      expire_date,  // format: YYYY-MM-DD
      keepdns
    } = req.body || {};

    const packages = getPackages();

    if (!domain || !username || !password || !email || !pkg || !expire_date) {
      return res.status(400).json({ ok: false, message: "Data belum lengkap." });
    }
    if (!isValidDomain(domain)) {
      return res.status(400).json({ ok: false, message: "Domain tidak valid." });
    }
    if (!isValidUsername(username)) {
      return res.status(400).json({ ok: false, message: "Username cPanel tidak valid (1-16, huruf/angka, mulai huruf)." });
    }
    if (!isValidEmail(email)) {
      return res.status(400).json({ ok: false, message: "Email tidak valid." });
    }
    if (!isStrongPassword(password)) {
      return res.status(400).json({ ok: false, message: "Password kurang kuat (min 10, ada huruf besar+kecil+angka)." });
    }
    if (!packages.includes(pkg)) {
      return res.status(400).json({ ok: false, message: "Package tidak dikenali." });
    }

    // Expired: end of day Asia/Makassar
    const exp = DateTime.fromISO(String(expire_date), { zone: "Asia/Makassar" });
    if (!exp.isValid) {
      return res.status(400).json({ ok: false, message: "Tanggal expired tidak valid." });
    }
    const expiresAtLocal = exp.endOf("day");
    const nowLocal = DateTime.now().setZone("Asia/Makassar");
    if (expiresAtLocal <= nowLocal) {
      return res.status(400).json({ ok: false, message: "Tanggal expired harus di masa depan." });
    }

    const keepdnsFinal = Number.isFinite(Number(keepdns))
      ? Number(keepdns)
      : Number(process.env.DEFAULT_KEEPDNS || 0);

    // Create via WHM
    const whm = await createCpanelAccount({ domain, username, password, email, pkg });
    const parsed = parseWhmCreateResult(whm);

    const status = parsed.ok ? "SUCCESS" : "FAILED";
    const message = parsed.message;

    // Simpan order (TIDAK simpan password)
    insertOrder({
      domain,
      username,
      email,
      pkg,
      expires_at: expiresAtLocal.toUTC().toISO(),
      keepdns: keepdnsFinal,
      status,
      whm_message: message,
      whm_raw: whm
    });

    if (!parsed.ok) {
      return res.status(500).json({ ok: false, message, whm });
    }

    // Output data cPanel sesuai request kamu
    const cpanelLink = `https://${domain}:2083`;
    return res.json({
      ok: true,
      message,
      cpanel: {
        domain,
        username,
        password,
        link: cpanelLink
      },
      expires_at: expiresAtLocal.toISO(),           // lokal Makassar (ISO)
      expires_at_utc: expiresAtLocal.toUTC().toISO()
    });

  } catch (err) {
    const msg = err?.response?.data ? "WHM error." : (err?.message || "Server error.");
    return res.status(500).json({
      ok: false,
      message: msg,
      detail: err?.response?.data || null
    });
  }
});

// --- Admin orders
app.get("/admin/orders", (req, res) => {
  const pass = req.query.pass || "";
  if (pass !== process.env.ADMIN_PASSWORD) {
    return res.status(401).send("Unauthorized");
  }

  const items = listOrders(400);
  const html = `
  <html>
    <head>
      <meta charset="utf-8"/>
      <meta name="viewport" content="width=device-width,initial-scale=1"/>
      <title>Hiro Hosting - Orders</title>
      <style>
        body{font-family:system-ui;background:#0b0f19;color:#e8f1ff;padding:16px}
        table{width:100%;border-collapse:collapse;background:#0f1629;border-radius:12px;overflow:hidden}
        th,td{padding:10px;border-bottom:1px solid rgba(255,255,255,.08);text-align:left;font-size:13px;vertical-align:top}
        th{background:#101b33}
        .ok{color:#4df0c6;font-weight:800}
        .bad{color:#ff6b81;font-weight:800}
        .muted{opacity:.7}
        code{font-size:12px}
      </style>
    </head>
    <body>
      <h2>Orders (latest)</h2>
      <p class="muted">Akses: <code>/admin/orders?pass=PASSWORD</code></p>
      <table>
        <thead>
          <tr>
            <th>ID</th>
            <th>Created</th>
            <th>Domain</th>
            <th>User</th>
            <th>Email</th>
            <th>Pkg</th>
            <th>Expires (UTC)</th>
            <th>Status</th>
            <th>WHM Msg</th>
            <th>Deleted</th>
            <th>Del Status</th>
            <th>Del Msg</th>
          </tr>
        </thead>
        <tbody>
          ${items.map(o => `
            <tr>
              <td>${o.id}</td>
              <td class="muted">${escapeHtml(o.created_at)}</td>
              <td>${escapeHtml(o.domain)}</td>
              <td>${escapeHtml(o.username)}</td>
              <td>${escapeHtml(o.email)}</td>
              <td>${escapeHtml(o.pkg)}</td>
              <td class="muted">${escapeHtml(o.expires_at)}</td>
              <td class="${o.status==="SUCCESS"?"ok":"bad"}">${escapeHtml(o.status)}</td>
              <td class="muted">${escapeHtml(o.whm_message || "")}</td>
              <td class="muted">${escapeHtml(o.deleted_at || "")}</td>
              <td class="${o.delete_status==="DELETED"?"ok":(o.delete_status?"bad":"")}">${escapeHtml(o.delete_status || "")}</td>
              <td class="muted">${escapeHtml(o.delete_message || "")}</td>
            </tr>
          `).join("")}
        </tbody>
      </table>
    </body>
  </html>`;

  res.setHeader("content-type", "text/html; charset=utf-8");
  res.send(html);
});

function escapeHtml(str) {
  return String(str)
    .replaceAll("&","&amp;")
    .replaceAll("<","&lt;")
    .replaceAll(">","&gt;")
    .replaceAll('"',"&quot;")
    .replaceAll("'","&#039;");
}

// --- Worker auto delete
async function runExpiryWorkerOnce() {
  const nowUtcIso = DateTime.now().toUTC().toISO();
  const targets = getExpiredUndeletedOrders(nowUtcIso, 50);
  if (!targets.length) return;

  for (const o of targets) {
    try {
      const whm = await removeCpanelAccount({ username: o.username, keepdns: o.keepdns });
      const parsed = parseWhmDeleteResult(whm);

      if (parsed.ok) {
        markDeleted(o.id, { delete_status: "DELETED", delete_message: parsed.message });
      } else {
        markDeleted(o.id, { delete_status: "DELETE_FAILED", delete_message: parsed.message });
      }
    } catch (e) {
      const reason = e?.response?.data ? "WHM delete error." : (e?.message || "Delete error.");
      markDeleted(o.id, { delete_status: "DELETE_FAILED", delete_message: reason });
    }
  }
}

const port = Number(process.env.PORT || 8080);
app.listen(port, () => {
  console.log(`Hiro Hosting running on http://localhost:${port}`);
});

// interval worker
const intervalMin = Number(process.env.WORKER_INTERVAL_MINUTES || 5);
setInterval(() => {
  runExpiryWorkerOnce().catch(() => {});
}, intervalMin * 60 * 1000);

// run once on start
runExpiryWorkerOnce().catch(() => {});