import Database from "better-sqlite3";

const db = new Database("data.sqlite");
db.pragma("journal_mode = WAL");

db.exec(`
CREATE TABLE IF NOT EXISTS orders (
  id INTEGER PRIMARY KEY AUTOINCREMENT,
  created_at TEXT NOT NULL,

  domain TEXT NOT NULL,
  username TEXT NOT NULL,
  email TEXT NOT NULL,
  pkg TEXT NOT NULL,

  expires_at TEXT NOT NULL,         -- UTC ISO
  keepdns INTEGER NOT NULL DEFAULT 0,

  status TEXT NOT NULL,             -- SUCCESS / FAILED
  whm_message TEXT,
  whm_raw TEXT,

  deleted_at TEXT,                  -- UTC ISO
  delete_status TEXT,               -- DELETED / DELETE_FAILED
  delete_message TEXT
);
`);

export function insertOrder({
  domain, username, email, pkg,
  expires_at, keepdns,
  status, whm_message, whm_raw
}) {
  const stmt = db.prepare(`
    INSERT INTO orders
    (created_at, domain, username, email, pkg, expires_at, keepdns, status, whm_message, whm_raw)
    VALUES
    (@created_at, @domain, @username, @email, @pkg, @expires_at, @keepdns, @status, @whm_message, @whm_raw)
  `);

  return stmt.run({
    created_at: new Date().toISOString(),
    domain,
    username,
    email,
    pkg,
    expires_at,
    keepdns: Number(keepdns || 0),
    status,
    whm_message: whm_message || null,
    whm_raw: whm_raw ? JSON.stringify(whm_raw) : null
  });
}

export function listOrders(limit = 300) {
  return db.prepare(`SELECT * FROM orders ORDER BY id DESC LIMIT ?`).all(limit);
}

export function getExpiredUndeletedOrders(nowUtcIso, limit = 50) {
  return db.prepare(`
    SELECT * FROM orders
    WHERE status='SUCCESS'
      AND deleted_at IS NULL
      AND expires_at <= ?
    ORDER BY id ASC
    LIMIT ?
  `).all(nowUtcIso, limit);
}

export function markDeleted(id, { delete_status, delete_message }) {
  return db.prepare(`
    UPDATE orders
    SET deleted_at=@deleted_at,
        delete_status=@delete_status,
        delete_message=@delete_message
    WHERE id=@id
  `).run({
    id,
    deleted_at: new Date().toISOString(),
    delete_status,
    delete_message: delete_message || null
  });
}